#ifndef BLITZSPEECH_H
#define BLITZSPEECH_H

#include <windows.h>

// Calling convention for the DLL
#define SPEECH_CALL __stdcall

// ========== SPEECH FLAGS ==========
#define SPF_DEFAULT 0
#define SPF_ASYNC 1
#define SPF_PURGEBEFORESPEAK 2
#define SPF_IS_FILENAME 4
#define SPF_IS_XML 8
#define SPF_IS_NOT_XML 16
#define SPF_PERSIST_XML 32
#define SPF_NLP_SPEAK_PUNC 64
#define SPF_PARSE_SAPI 128
#define SPF_PARSE_SSML 256

// ========== AUDIO FORMAT CONSTANTS ==========
// Sample Rates (Hz)
#define SP_SAMPLE_RATE_8000 8000
#define SP_SAMPLE_RATE_11025 11025
#define SP_SAMPLE_RATE_16000 16000
#define SP_SAMPLE_RATE_22050 22050
#define SP_SAMPLE_RATE_44100 44100
#define SP_SAMPLE_RATE_48000 48000

// Bit Depths
#define SP_BIT_DEPTH_8 8
#define SP_BIT_DEPTH_16 16

// Channels
#define SP_CHANNEL_MONO 1
#define SP_CHANNEL_STEREO 2

// ========== LANGUAGE CONSTANTS ==========
#define SP_LAN_ENGLISH "English"
#define SP_LAN_SPANISH "Spanish"
#define SP_LAN_FRENCH "French"
#define SP_LAN_GERMAN "German"
#define SP_LAN_ITALIAN "Italian"
#define SP_LAN_PORTUGUESE "Portuguese"
#define SP_LAN_DUTCH "Dutch"
#define SP_LAN_RUSSIAN "Russian"
#define SP_LAN_POLISH "Polish"
#define SP_LAN_CHINESE "Chinese"
#define SP_LAN_CHINESE_SIMPLIFIED "Chinese (Simplified)"
#define SP_LAN_CHINESE_TRADITIONAL "Chinese (Traditional)"
#define SP_LAN_JAPANESE "Japanese"
#define SP_LAN_KOREAN "Korean"
#define SP_LAN_ENGLISH_US "English (United States)"
#define SP_LAN_ENGLISH_UK "English (Great Britain)"
#define SP_LAN_ENGLISH_GB "English (Great Britain)"
#define SP_LAN_ENGLISH_AU "English (Australia)"
#define SP_LAN_ENGLISH_CA "English (Canada)"
#define SP_LAN_ENGLISH_IN "English (India)"
#define SP_LAN_ARABIC "Arabic"
#define SP_LAN_CATALAN "Catalan"
#define SP_LAN_CZECH "Czech"
#define SP_LAN_DANISH "Danish"
#define SP_LAN_FINNISH "Finnish"
#define SP_LAN_GREEK "Greek"
#define SP_LAN_HEBREW "Hebrew"
#define SP_LAN_HINDI "Hindi"
#define SP_LAN_HUNGARIAN "Hungarian"
#define SP_LAN_NORWEGIAN "Norwegian"
#define SP_LAN_SWEDISH "Swedish"
#define SP_LAN_THAI "Thai"
#define SP_LAN_TURKISH "Turkish"

// ========== VOICE TYPES ==========
#define SP_VOICE_TYPE_ERROR -1
#define SP_VOICE_TYPE_DESKTOP 0
#define SP_VOICE_TYPE_ONECORE 1

// ========== ERROR CODES ==========

// InitSpeech() returns:
#define SP_INIT_SUCCESS 0
#define SP_INIT_ERR_COM_FAILED -1           // COM initialization failed
#define SP_INIT_ERR_VOICE_FAILED -2         // Failed to create voice instance

// SpeakText() returns:
#define SP_SPEAK_SUCCESS 0
#define SP_SPEAK_ERR_NOT_INIT -1            // Not initialized
#define SP_SPEAK_ERR_NULL_TEXT -2           // NULL text pointer
#define SP_SPEAK_ERR_CONV_LEN -3            // Failed to get wide string length
#define SP_SPEAK_ERR_CONV_ALLOC -4          // Failed to allocate conversion buffer
#define SP_SPEAK_ERR_CONV_FAILED -5         // String conversion failed
#define SP_SPEAK_ERR_FAILED -6              // SAPI Speak call failed

// SpeakTextFlags() returns: (same as SpeakText)
#define SP_SPEAK_FLAGS_SUCCESS 0
#define SP_SPEAK_FLAGS_ERR_NOT_INIT -1
#define SP_SPEAK_FLAGS_ERR_NULL_TEXT -2
#define SP_SPEAK_FLAGS_ERR_CONV_LEN -3
#define SP_SPEAK_FLAGS_ERR_CONV_ALLOC -4
#define SP_SPEAK_FLAGS_ERR_CONV_FAILED -5
#define SP_SPEAK_FLAGS_ERR_FAILED -6

// SpeakFile() returns:
#define SP_FILE_SUCCESS 0
#define SP_FILE_ERR_NOT_INIT -1             // Not initialized
#define SP_FILE_ERR_NULL_PATH -2            // NULL filepath
#define SP_FILE_ERR_CANT_OPEN -3            // File not found or can't open
#define SP_FILE_ERR_TOO_LARGE -4            // File too large (>10MB) or empty
#define SP_FILE_ERR_ENCODING -5             // Encoding conversion failed
#define SP_FILE_ERR_SPEAK_FAILED -6         // SAPI Speak call failed

// GetFileEncoding() returns:
#define SP_ENC_ANSI 0                       // ANSI/default code page
#define SP_ENC_UTF16_LE 1                   // UTF-16 Little Endian
#define SP_ENC_UTF16_BE 2                   // UTF-16 Big Endian
#define SP_ENC_UTF8 3                       // UTF-8 (with or without BOM)
#define SP_ENC_ERR_NULL_PATH -1             // NULL filepath

// PauseSpeech() returns:
#define SP_PAUSE_SUCCESS 0
#define SP_PAUSE_ERR_NOT_INIT -1            // Not initialized
#define SP_PAUSE_ERR_FAILED -2              // SAPI Pause call failed

// ResumeSpeech() returns:
#define SP_RESUME_SUCCESS 0
#define SP_RESUME_ERR_NOT_INIT -1           // Not initialized
#define SP_RESUME_ERR_FAILED -2             // SAPI Resume call failed

// StopSpeech() returns:
#define SP_STOP_SUCCESS 0
#define SP_STOP_ERR_NOT_INIT -1             // Not initialized
#define SP_STOP_ERR_FAILED -2               // SAPI Stop call failed

// IsSpeaking() returns:
#define SP_IS_SPEAKING_YES 1                // Currently speaking
#define SP_IS_SPEAKING_NO 0                 // Not speaking
#define SP_IS_SPEAKING_ERR -1               // Error/not initialized

// WaitUntilSpeechIsDone() returns:
#define SP_WAIT_SUCCESS 0
#define SP_WAIT_ERR_NOT_INIT -1             // Not initialized
#define SP_WAIT_ERR_FAILED -2               // Wait failed or timed out

// SetSpeechRate() returns:
#define SP_RATE_SUCCESS 0
#define SP_RATE_ERR_NOT_INIT -1             // Not initialized
#define SP_RATE_ERR_FAILED -2               // SAPI SetRate call failed

// SetSpeechVolume() returns:
#define SP_VOLUME_SUCCESS 0
#define SP_VOLUME_ERR_NOT_INIT -1           // Not initialized
#define SP_VOLUME_ERR_FAILED -2             // SAPI SetVolume call failed

// GetVoiceCount() returns:
// Positive number = voice count
#define SP_VOICE_COUNT_ERR_NOT_INIT -1      // Not initialized
#define SP_VOICE_COUNT_ERR_ENUM_FAILED -2   // Failed to enumerate voices
#define SP_VOICE_COUNT_ERR_GET_COUNT -3     // Failed to get count

// GetCurrentVoice() returns:
// 0 or positive = current voice index
#define SP_GET_VOICE_ERR_NOT_INIT -1        // Not initialized
#define SP_GET_VOICE_ERR_NO_VOICE -2        // Failed to get current voice
#define SP_GET_VOICE_ERR_NO_ID -3           // Failed to get voice ID
#define SP_GET_VOICE_ERR_NO_ENUM -4         // Voice enumeration not done
#define SP_GET_VOICE_ERR_NOT_FOUND -5       // Current voice not in list

// SetSpeechVoice() returns:
#define SP_SET_VOICE_SUCCESS 0
#define SP_SET_VOICE_ERR_NOT_INIT -1        // Not initialized
#define SP_SET_VOICE_ERR_BAD_INDEX -2       // Invalid voice index
#define SP_SET_VOICE_ERR_GET_TOKEN -3       // Failed to get voice token
#define SP_SET_VOICE_ERR_FAILED -4          // SAPI SetVoice call failed

// SetSpeechVoiceByName() returns:
// 0 or positive = voice index (success)
#define SP_SET_NAME_ERR_NOT_INIT -1         // Not initialized or NULL name
#define SP_SET_NAME_ERR_ENUM_FAILED -2      // Failed to enumerate voices
#define SP_SET_NAME_ERR_SET_FAILED -3       // Found voice but SetVoice failed
#define SP_SET_NAME_ERR_NOT_FOUND -4        // Voice name not found

// SpeakToFile() and SpeakToFileDefault() return:
#define SP_FILE_OUT_SUCCESS 0
#define SP_FILE_OUT_ERR_NOT_INIT -1         // Not initialized
#define SP_FILE_OUT_ERR_NULL_PARAM -2       // NULL text or filepath
#define SP_FILE_OUT_ERR_TEXT_LEN -3         // Failed to get text length
#define SP_FILE_OUT_ERR_TEXT_CONV -4        // Text conversion failed
#define SP_FILE_OUT_ERR_PATH_LEN -5         // Failed to get path length
#define SP_FILE_OUT_ERR_PATH_CONV -6        // Path conversion failed
#define SP_FILE_OUT_ERR_CREATE_STREAM -7    // Failed to create stream
#define SP_FILE_OUT_ERR_BIND_FILE -8        // Failed to bind to file
#define SP_FILE_OUT_ERR_SET_OUTPUT -9       // Failed to set output
#define SP_FILE_OUT_ERR_SPEAK_FAILED -10    // Speak call failed
#define SP_FILE_OUT_ERR_INVALID_RATE -11    // Invalid sample rate
#define SP_FILE_OUT_ERR_INVALID_BITS -12    // Invalid bit depth
#define SP_FILE_OUT_ERR_INVALID_CHANNELS -13 // Invalid channel count

// ========== FUNCTION DECLARATIONS ==========
#ifdef __cplusplus
extern "C" {
#endif

    // Core initialization
    __declspec(dllimport) int SPEECH_CALL InitSpeech();
    __declspec(dllimport) void SPEECH_CALL CleanupSpeech();

    // Basic speech functions
    __declspec(dllimport) int SPEECH_CALL SpeakText(const char* text);
    __declspec(dllimport) int SPEECH_CALL SpeakTextFlags(const char* text, int flags);
    __declspec(dllimport) int SPEECH_CALL SpeakFile(const char* filepath);

    // Speech control
    __declspec(dllimport) int SPEECH_CALL PauseSpeech();
    __declspec(dllimport) int SPEECH_CALL ResumeSpeech();
    __declspec(dllimport) int SPEECH_CALL StopSpeech();
    __declspec(dllimport) int SPEECH_CALL IsSpeaking();
    __declspec(dllimport) int SPEECH_CALL WaitUntilSpeechIsDone(int timeout_ms);

    // Voice settings
    __declspec(dllimport) int SPEECH_CALL SetSpeechRate(int rate);
    __declspec(dllimport) int SPEECH_CALL SetSpeechVolume(int volume);

    // Voice management
    __declspec(dllimport) int SPEECH_CALL GetVoiceCount();
    __declspec(dllimport) const char* SPEECH_CALL GetVoiceName(int index);
    __declspec(dllimport) const char* SPEECH_CALL GetVoiceLanguage(int index);
    __declspec(dllimport) int SPEECH_CALL GetVoiceType(int index);
    __declspec(dllimport) int SPEECH_CALL GetCurrentVoice();
    __declspec(dllimport) int SPEECH_CALL SetSpeechVoice(int index);
    __declspec(dllimport) int SPEECH_CALL SetSpeechVoiceByName(const char* name);

    // File output
    __declspec(dllimport) int SPEECH_CALL SpeakToFileDefault(const char* text, const char* filepath);
    __declspec(dllimport) int SPEECH_CALL SpeakToFile(const char* text, const char* filepath,
        int sampleRate, int bitsPerSample, int channels, int flags);

    // Utilities
    __declspec(dllimport) int SPEECH_CALL GetFileEncoding(const char* filepath);
    __declspec(dllimport) void SPEECH_CALL AboutBlitzSpeech();

#ifdef __cplusplus
}
#endif

#endif // BLITZSPEECH_H